﻿param ([int] $keepdays = 0, [int] $keepmonths = 1, $datadir = “”, $dryrun = 1);
if ($dryrun) {
    Write-Host “DRYRUN - data isn’t actually being deleted.”
} else {
    Write-Host “LIVE - data is being deleted.”
}
# Figure out the date beforehand that it is OK to delete:
#
# -keepdays   Is in the format of days to keep. For example, on monthly flexstore, a value of 30 days
#             won’t delete the month of January on March 1st because #January 31st is less than 30 days away.
#
# -keepmonths Is in the format of months to keep. On monthly flexstore, 1 month will always keep at least 1
#             full month, so January would be deleted on March 1st, but February won’t be deleted
#             until there’s a complete full month available - February would be deleted on April 1st though
#             (March is complete as well)
#
# -datadir    If a custom data directory is in use, this is where the data will be deleted from. 
#             Should *not* include the “SampleSets” part of the directory (just the root storage directory).          
#
# -dryrun     Just show what’s going to be deleted and don’t actually delete anything. Defaults to 1. 
#             Change the script or pass 0 as a parameter to actually delete data.
#
if ($keepdays -gt 0) {
    $keepdate = (Get-Date).AddDays(-$keepdays);
} elseif ($keepmonths -gt 0) {
    $keepdate = (Get-Date).AddMonths(-$keepmonths);
} else {
    throw “Both keepdays and keepmonths are 0 or negative. Must specify something to delete.”
}
$keepdate = $keepdate.ToUniversalTime();
Write-Host “Deleting data files that only contain data collected before: $keepdate UTC”
#
# If datadir is passed in, then use that. Otherwise, check if running as a service and use that.
#
# It’s possible that a data file is open and if that’s the case it won’t be deletable. It should be
# deleted next time through. Open files have recently had something done to them, but they get closed
# after a few minutes of access.
#
# Get a list of all directories that may be storing data in:
# As application: %localappdata%\PingPlotter 5\SampleSets
# As service: %programdata%\PingPlotter 5\SampleSets”
#
$folders = @();
if ($datadir) {
    $folders += $datadir;
} else {
    if (Get-Service “PingPlotter5” -ErrorAction SilentlyContinue) {
        $folders += ($env:ProgramData + “\PingPlotter 5")
        Write-Host “PingPlotter is installed as a service.”
    } else {
        $folders += ($env:LocalAppData + “\PingPlotter 5")
        Write-Host “PingPlotter is installed as an application, not a service.”
    }
}
#
# For customization, you can add more directories here, but PingPlotter really only collects data
# in a single folder.
#
Foreach ($folder in $folders)
{
    $folder = Join-Path $folder “SampleSets”
    Write-Host “Deleting in $folder”;
    #
    # Note: All file dates are UTC, not local time. Timestamps on files are local time, so they may not match
    # exactly. Also, files may be written to after no more data is being collected in to it - in the case
    # of a comment being created in the past, or if aggregation is run (or rerun) for some reason.
    #
    # Iterate all date-looking directories in that folder. Date-looking directories are in this format:
    # * Annual flexstore yyyy
    # * Monthly flexstore yyyy-mm
    # * Daily flexstore yyyy-mm-dd
    # Check the most recent piece of data that could be stored in this folder (end of each period)
    # If the date is before “Keep” date, delete it.
    #
    $subfolders = Get-ChildItem -Directory $folder
    Foreach ($deletable in $subfolders) {
        $deletable = $deletable.Name
        if ($deletable -match “^(\d\d\d\d)-(\d\d)-(\d\d)$“) {
            # Daily file.
            $folderDate = (Get-Date -Year $Matches[1] -Month $Matches[2] -Day $Matches[3]).Date.AddDays(1).AddMilliseconds(-1)
        } elseif ($deletable -match “^(\d\d\d\d)-(\d\d)$“) {
            # Monthly file.
            $folderDate = (Get-Date -Year $Matches[1] -Month $Matches[2] -Day 1).Date.AddMonths(1).AddMilliseconds(-1)
        } elseif ($deletable -match “^(\d\d\d\d)$“) {
            # Annual file.
            $folderDate = (Get-Date -Year $Matches[1] -Month 1 -Day 1).Date.AddYears(1).AddMilliseconds(-1)
        } else {
            Write-Host “$deletable isn’t a date folder”
            continue;
        }
        if ($folderDate -lt $keepdate) {
            Write-Host “$deletable holds OLD data before $folderDate UTC and can be deleted”;
            if (!$dryrun) {
                $dirToDelete = Join-Path $folder $deletable
                Write-Host “Deleting $dirToDelete”;
                Remove-Item -Path $dirToDelete -Recurse
            }
        } else {
            Write-Host “$deletable holds NEW data and will be kept ($folderDate) UTC”
        }
    }
}